/* ///////////////////////////////////////////////////////////////////////// */
/*  This is part of the source of the OMAP 5912 heterogeneous dual-core      */
/*  MPEG-4 SP video decoder published in ACM Transactions on Embedded        */
/*  Computing Systems, Vol. X, Issue Y.                                      */
/* ------------------------------------------------------------------------- */
/*  The source code is released under GPL license.                           */
/*                                                                           */
/*  Copyright, 2011                                                          */
/*  Multimedia Embedded Systems Labs                                         */
/*  Dept. of Computer Science                                                */
/*  National Chiao Tung University                                           */
/*  Hsinchu, Taiwan.                                                         */
/* ///////////////////////////////////////////////////////////////////////// */

#include <stdio.h>

#include <csl_intc.h>
#include <csl_mbx.h>
#include <soc.h>

#include "metypes.h"
#include "mbx_command.h"
#include "mem_address.h"
#include "dual_core_config.h"

#define MPU_READ_TIMER1 *(uint32*)0xFFFEC508

xint arm_announce(uint16 command, uint16 data);

#define MAX(a,b) ((a)>(b)?(a):(b))

volatile uint16 end_of_free_decoder;
volatile uint16 dsp_ready = 0;

extern int slice_number_per_frame;
extern volatile int arm_do;
extern volatile int dsp_do;
extern volatile int not_finishing_decoding_frame;
extern int is_pframe;
extern volatile int dsp_display_ok;

extern uint32 task_dispatch_isr_overhead;
extern uint32 task_dispatch_isr_counter;

CSL_MbxHandle hMbx;
int dsp_decode_slice_number = 0;

void
task_dispatch_isr(void *arg)
{
    CSL_Status status;
    CSL_MbxObj mbxObj;
    CSL_MbxParam mbxParam;
    CSL_MbxData mbxData;
    uint16  command;
    uint32 time_count;

    //  printf("DSP2ARM mbx1 interrupt !\n");
    time_count = MPU_READ_TIMER1;
    task_dispatch_isr_counter++;

    mbxData.dataWidth = CSL_MBX_DATA_32BIT;
    if (CSL_SOK != CSL_mbxGetHwStatus(hMbx, CSL_MBX_QUERY_DATAREGS, &mbxData))
    {
        printf("task_dispatch_isr error !\n");
        CSL_mbxClose(hMbx);
        exit();
    }

    //  printf("Data received from DSP [%#lx]\n", mbxData.data);
    command = (uint16) (mbxData.data >> 16);
    // data = (uint16) (mbxData.data & 0x0000FFFF);

    switch (command)
    {
    case D2A_READY:
        dsp_ready = 1;
        break;

    case D2A_INITIALIZATION_DONE:
#ifndef ARM_ONLY
        arm_announce(A2D_INITIALIZE_TASK, *((uint16 *) Bistream_address));
#endif
        break;

    case D2A_TASK_INITIALIZATION_DONE:
        if (not_finishing_decoding_frame)
        {
            dsp_do = MAX(arm_do, dsp_do) + 1;
            if (dsp_do < slice_number_per_frame)
            {
                arm_announce(A2D_DECODE_SLICE, dsp_do + is_pframe);
                dsp_decode_slice_number++;
            }
        }
        break;

    case D2A_DECODING_SLICE_DONE:
        if (dsp_do == slice_number_per_frame - 1)
        {
            not_finishing_decoding_frame = 0;
        }

        if (not_finishing_decoding_frame)
        {
            dsp_do = MAX(arm_do, dsp_do) + 1;
            if (dsp_do < slice_number_per_frame)
            {
                arm_announce(A2D_DECODE_SLICE, dsp_do + is_pframe);
                dsp_decode_slice_number++;
            }
        }
        break;

    case D2A_RELEASING_DONE:
        end_of_free_decoder = 1;
        break;

    case D2A_DISPLAY_DONE:
        dsp_display_ok = 1;
        break;

    default:
        printf("TASK_DISPATCH_ISR Error command from DSP: %d\n", command);
    }

    /* reopen the mailbox instance for next reception */
    hMbx = CSL_mbxOpen(&mbxObj, CSL_MBX_DSP2ARM1, &mbxParam, &status);
    if (CSL_SOK != status)
    {
        printf("Error opening mailbox(DSP2ARM1) instance !\n");
        exit();
    }

	task_dispatch_isr_overhead += (time_count - MPU_READ_TIMER1);
}

extern void _CSL_intcDispatchFIQ();

xint
listen_dsp()
{
    CSL_Status status;
    CSL_IntcObj intcObj;
    CSL_IntcHandle hIntc;
    CSL_IntcEventHandlerRecord isrRec;

    CSL_MbxObj mbxObj;
    CSL_MbxParam mbxParam;

    CSL_IntcHwSetup intcHwSetup;
    CSL_IntcEventEnableState intcState;

    /* Initializa the mailbox CSL */
    CSL_mbxInit(NULL);

    /* Open the mailbox instance for reception */
    hMbx = CSL_mbxOpen(&mbxObj, CSL_MBX_DSP2ARM1, &mbxParam, &status);
    if (CSL_SOK != status)
    {
        printf("Error opening mailbox(DSP2ARM1) instance !\n");
        exit();
    }

    /* Obtain a handle to the Interrupt module corresponding to dsp2arm1 mbx instance */
    hIntc = CSL_intcOpen(&intcObj, CSL_INTC_EVENTID_DSPMBX1, NULL, &status);
    if (status != CSL_SOK)
    {
        printf("Interrupt Handler (DSP2ARM1) open failed !\n");
        CSL_mbxClose(hMbx);
        exit();
    }

    isrRec.handler = task_dispatch_isr;

    /* Configure the interrupt controller module */
    intcHwSetup.priority = CSL_INTC_PRIORITY_DEFAULT;
    intcHwSetup.sense = CSL_INTC_SENSE_ACTIVE_LOW;
    intcHwSetup.type = CSL_INTC_TYPE_FIQ;

    CSL_intcHwSetup(hIntc, &intcHwSetup);
    CSL_intcPlugEventHandler(hIntc, &isrRec);
    CSL_intcEventEnable(CSL_INTC_EVENTID_DSPMBX1, &intcState);
    *(uint32 *) 0x00000038 = (uint32) _CSL_intcDispatchFIQ;
    return 0;
}
